/**
 * \file: exchnd_collector.c
 *
 * Collector implementation
 * Some data collected in the exception handler is collected in the user
 * space. The file defines list of collectors and the implementation of
 * the collectors.
 *
 * \component: Exception handler
 *
 * \author: M.Weise
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 * \history
 * 2013/12/05 Initial version
 *
 ***********************************************************************/
#define _GNU_SOURCE

#include <ctype.h>
#include <dirent.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <fts.h>
#include <limits.h>
#include <stdarg.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <linux/exchnd.h>

#include <sys/ioctl.h>
#include <sys/ptrace.h>
#include <sys/stat.h>
#include <sys/statvfs.h>

#include "exchnd_collector.h"

#define NS_PER_S 1000000000

#define LABEL_BACKTRACE "====== backtrace:"
#define LABEL_FB_BACKTRACE "====== frame based backtrace:"
#define LABEL_EXT_BACKTRACE "====== extended backtrace:"
#define LABEL_GUESS_BACKTRACE "====== functions in stack:"
#define LABEL_BACKTRACE_ALL_THREADS "====== backtrace for all threads:"
#define LABEL_FS_STATE "====== file system status:"
#define LABEL_CG_STATE "====== cgroup status:"
#define LABEL_MEMORYMAP "====== memory map:"
#define LABEL_STACK_DUMP "====== stack dump:"
#define LABEL_MEM_DUMP "====== memory dump:"
#define LABEL_PROCESS_LIST "====== process list:"
#define TABLE_PROCESS_LIST  \
    "PID   STATUS  PRI  NICE RT_PRI POLICY " \
    "     RSS (Pages) " \
    "          SWAPENTS " \
    "              COMM Command line"
#define TMPL_PROCESS_LIST \
    "%-5s %6s %4s %5s %6s %6s %16s %16lukB %18s %s"

#define LABEL_THREAD_LIST   "====== thread list:"
#define TABLE_THREAD_LIST   "PID   (PGRP ) STATUS COMM"
#define TMPL_THREAD_LIST    "%-5s (%-5s) %6s %-16s"

#define LABEL_APPLICATION_SPECIFIC "====== application specific:"
#define LABEL_CPU_USAGE "====== cpu usage:"

#define PROCESS_LINE_STARVED    0
#define PROCESS_LINE_FULL       1
#define PROCESS_LINE_THREAD     2
#define LABEL_CMDLINE "Command line: %s"

#define PROCFS "/proc"
#define MOUNTS "/proc/mounts"

#define TMPL_MAPS "/proc/%u/maps"
#define TMPL_CMDLINE "/proc/%s/cmdline"
#define TMPL_STAT "/proc/%d/stat"
#define TMPL_STATUS "/proc/%s/status"
#define TMPL_TASK "/proc/%d/task"
#define TMPL_TASK_STAT "/proc/%d/task/%d/stat"
#define TMPL_CG_PERCPU "%s/cpuacct.usage_percpu"

#define CG_INFO_FILE "/proc/cgroups"
#define CG_SUB_CPUACCT "cpuacct"
#define CG_SUB_MEMORY "memory"
#define CG_PROCS "cgroup.procs"
#define CG_MEM_USAGE "memory.usage_in_bytes"
#define CG_MEM_MAX_USAGE "memory.max_usage_in_bytes"
#define CG_MEM_STAT "memory.stat"

#define MEMSTAT_CACHE "cache"
#define MEMSTAT_RSS "rss"
#define MEMSTAT_SWAP "swap"

enum stat_type {
    STAT_PID = 0,
    STAT_COMM = 1,
    STAT_STATE = 2,
    STAT_PGRP = 4,
    STAT_UTIME = 13,
    STAT_STIME = 14,
    STAT_PRIORITY = 17,
    STAT_NICE = 18,
    STAT_STARTTIME = 21,
    STAT_RSS = 23,
    STAT_RTPRIORITY = 39,
    STAT_POLICY = 40,
    STAT_MAX = 52
};

#define STR_INDENT "  "

#ifndef EXCHND_CMDLINE_LEN
#   define EXCHND_CMDLINE_LEN 4096
#endif

long attached[MAX_THREADS];
int idx_max = 0;

/* Collectors */

/*
 * \func get_maps
 *
 * Helper to store content of maps for a process.
 * Only useful content is stored (ones with names).
 *
 * \param pid PID of the process to read the map of.
 * \param maps Pointer to the map that will be initialized by the function.
 *
 * \return Number of entries filled in.
 */
static int get_maps(int pid, struct exchnd_maps **maps)
{
    char filename[32] = { '\0' };
    char buffer[512] = { '\0' };
    struct exchnd_file maps_file;
    char *pos_end = NULL;
    char *pos_sep = NULL;
    char *end_ptr = NULL;
    char *pathname = NULL;
    unsigned long region_start;
    unsigned long region_end;
    int region_found = 0;
    int max_region = 16;

    if (!maps)
        return -EINVAL;

    snprintf(filename, sizeof(filename), TMPL_MAPS, pid);
    exchnd_init_file(&maps_file);
    maps_file.fd = open(filename, O_RDONLY);

    /* klockwork fix */
    if (maps_file.fd < 0) {
        region_found = -EINVAL;
    } else {
        /* First let's allocate a minimal buffer. */
        *maps = malloc(max_region * sizeof(struct exchnd_maps));

        if (!(*maps)) {
            exchnd_log_error("No mem to get maps");
            region_found = -ENOMEM;
        }

        /* Read /proc/<pid>/maps line wise */
        while (*maps && exchnd_fgets(buffer, sizeof(buffer), &maps_file)) {
            /* Mark the region separator (dash) */
            pos_sep = strchr(buffer, '-');
            /* Mark the end of the address region */
            pos_end = strchr(buffer, ' ');

            if (pos_end && pos_sep && (pos_sep < pos_end)) {
                region_start = strtoul(buffer, &end_ptr, BASE_HEX);
                region_end = strtoul(pos_sep + 1, &end_ptr, BASE_HEX);
            } else {
                /* Invalid entry ? */
                continue;
            }

            pathname = strrchr(buffer, ' ');

            /* Ignore empty fields */
            if (pathname) {
                pathname++;

                if (region_found >= max_region) {
                    /* Then double the size */
                    max_region *= 2;
                    *maps = realloc(*maps,
                                    max_region * sizeof(struct exchnd_maps));

                    if (!*maps) {
                        exchnd_log_error("No mem to get maps");
                        region_found = -ENOMEM;
                        break;
                    }
                }

                (*maps)[region_found].read = 0;
                (*maps)[region_found].write = 0;
                (*maps)[region_found].exec = 0;

                pos_end++;

                if (*pos_end++ == 'r')
                    (*maps)[region_found].read = 1;

                if (*pos_end++ == 'w')
                    (*maps)[region_found].write = 1;

                if (*pos_end++ == 'x')
                    (*maps)[region_found].exec = 1;

                strncpy((*maps)[region_found].name, pathname, 512);
                /* Remove the trailing newline */
                end_ptr = strrchr((*maps)[region_found].name, '\n');

                if (end_ptr)
                    *end_ptr = '\0';

                if ((strlen((*maps)[region_found].name) == 0) &&
                    (((*maps)[region_found].read == 1) ||
                     ((*maps)[region_found].write == 1) ||
                     ((*maps)[region_found].exec == 1)))
                    strncpy((*maps)[region_found].name, "[ANONYMOUS]", 512);

                (*maps)[region_found].start = region_start;
                (*maps)[region_found].end = region_end;

                region_found++;
            }
        }

        /* Closing the file */
        exchnd_close(&maps_file);
    }

    return region_found;
}

/*
 * \func remove_trailing_newline
 *
 * This function removes a trailing newline of a string.
 *
 * \param string The string where the newline shall be removed.
 */
static void remove_trailing_newline(char *string)
{
    int last_char = strlen(string) - 1;

    if (string[last_char] == '\n')
        string[last_char] = '\0';
}

/*
 * \func skip_word
 *
 * Helper for get_comm.
 *
 * \param p Pointer to current word
 *
 * \return Point to next word.
 */
static inline char *skip_word(const char *p)
{
    while (isspace(*p))
        p++;

    /* Skip comm, there may be spaces in the middle. */
    if (*p && (*p == '('))
        while (*p && (*p != ')'))
            p++;

    while (*p && !isspace(*p))
        p++;

    return (char *)p;
}

/*
 * \func copy_word
 *
 * Helper for collect_stat.
 *
 * \param p Pointer to current word
 * \param d Pointer to destination
 *
 * \return Pointer to next word.
 */
static inline char *copy_word(const char *p, char *d)
{
    while (isspace(*p))
        p++;

    /* Skip comm, there may be spaces in the middle. */
    if (*p && (*p == '('))
        while (*p && (*p != ')'))
            p++;

    while (*p && !isspace(*p))
        *d++ = *p++;

    return (char *)p;
}

static unsigned long collect_swapents(char *pid_str)
{
    struct exchnd_file file;
    char filename[32];
    char stat_raw[MAX_MSG_LEN];
    unsigned long ret = 0;

    /* Extract process info from /proc/<pid>/status */
    snprintf(filename, sizeof(filename), TMPL_STATUS, pid_str);

    exchnd_init_file(&file);
    file.fd = open(filename, O_RDONLY);

    if (file.fd < 0)
        return -1UL;

    /* Getting data from stat [pid (exec_name) status ...]*/
    while (exchnd_fgets(stat_raw, MAX_MSG_LEN, &file) != NULL) {
        char *p;

        if ((p = strstr(stat_raw, "VmSwap:")) != NULL) {
            p += sizeof("VmSwap:");

            ret = strtoul(p, NULL, BASE_DEC);
            break;
        }
    }

    exchnd_close(&file);

    return ret;
}

static void collect_stat(int pid, char *pid_str, char instat[STAT_MAX][64])
{
    struct exchnd_file file;
    char filename[32];
    char stat_raw[MAX_MSG_LEN];

    memset(instat, 0, sizeof(*instat));

    if (pid)
        /* Extract thread info from /proc/<pid>/task/tid/stat */
        snprintf(filename, sizeof(filename), TMPL_TASK_STAT, pid,
                 atoi(pid_str));
    else
        /* Extract process info from /proc/<pid>/stat */
        snprintf(filename, sizeof(filename), TMPL_STAT, atoi(pid_str));

    exchnd_init_file(&file);
    file.fd = open(filename, O_RDONLY);

    if (file.fd < 0)
        return;

    /* Getting data from stat [pid (exec_name) status ...]*/
    if (exchnd_fgets(stat_raw, MAX_MSG_LEN, &file)) {
        char *p = stat_raw;
        char *q = NULL;
        unsigned int i = 0;

        p = copy_word(p, instat[i++]); /* pid */

        p = memchr(p, '(', 512);

        /* get comm */
        if (!p)
            goto out;

        q = memchr(p, ')', 512);

        if (q && (q > p)) {
            strncpy(instat[i++], p + 1, (q - 1) - p);
            p = ++q;
        }

        for (; i < STAT_MAX; i++)
            p = copy_word(p, instat[i]);
    }

out:
    exchnd_close(&file);
}

/*
 * \func get_cpu_time
 *
 * Collects the cpu time consumed by a specific process.
 *
 */
static void get_cpu_time(int pid, char *pid_str, char *buffer)
{
    char s_cpu[64] = { '\0' };
    char s_run[64] = { '\0' };
    static long clock_ticks = 0;
    double uptime = 0;
    unsigned long t = 0;
    unsigned long start_time = 0;
    char s_data[STAT_MAX][64] = { '\0' };

    collect_stat(pid, pid_str, s_data);

    if (clock_ticks == 0) {
        clock_ticks = sysconf(_SC_CLK_TCK);

        if (clock_ticks == -1) {
            exchnd_print_error("Unable to get clock ticks.");
            return;
        }
    }

    t = strtoul(s_data[STAT_UTIME], NULL, BASE_DEC); /* utime */
    t += strtoul(s_data[STAT_STIME], NULL, BASE_DEC); /* stime */

    snprintf(s_cpu, 64, "(%lu) %.2f", t, (double)t / clock_ticks);

    start_time = strtoul(s_data[STAT_STARTTIME], NULL, BASE_DEC);

    {
        int fd;
        char buff[64] = { '\0' };

        if ((fd = open("/proc/uptime", 0)) != -1) {
            if (read(fd, buff, sizeof(buff)) > 0)
                uptime = strtoul(buff, NULL, BASE_DEC);

            close(fd);
        }
    }

    snprintf(s_run, 64, "%.2f", uptime - (double)start_time / clock_ticks);

    snprintf(buffer,
             MAX_MSG_LEN,
             "%5.10s\tCPU time: %15.64ss  Runtime:%15.64ss\t%s",
             pid_str,
             s_cpu,
             s_run,
             s_data[STAT_COMM]);

}

void get_comm(char *stat_raw, char *comm)
{
    char *p = stat_raw;
    char *q = NULL;

    memset(comm, '\0', 64);

    p = skip_word(p); /* pid */

    p = memchr(p, '(', 64);

    if (!p)
        return;

    q = memchr(p, ')', 64);

    if (q && (q > p))
        strncpy(comm, p + 1, (q - 1) - p);
}

/*
 * \func get_process_line
 *
 * This function generates a line for use with the process and thread list.
 * The output consists of the PID, the process state and the command line
 * of the given process.
 *
 * \param pid_str Process ID string
 * \param buffer Storage buffer for the line
 * \param header Size of the storage buffer
 *
 * \return 0 if successful. An error code otherwise
 */
static int get_process_line(char *pid_str, char *buffer, int size, int type)
{
    struct exchnd_file file;
    char filename[32];
    char *cmdline_raw;
    char s_data[STAT_MAX][64] = { '\0' };
    char *cmd_cursor;
    int len = 0;

    /* Don't use calloc as it generate minor page faults */
    cmdline_raw = malloc(EXCHND_CMDLINE_LEN * sizeof(char));

    if (!cmdline_raw) {
        exchnd_log_error("No space left for command line.");
        return -ENOMEM;
    }

    memset(cmdline_raw, 0, EXCHND_CMDLINE_LEN * sizeof(char));

    /* Read /proc/<pid>/cmdline */
    snprintf(filename, sizeof(filename), TMPL_CMDLINE, pid_str);
    exchnd_init_file(&file);
    file.fd = open(filename, O_RDONLY);

    /* klockwork fix */
    if (file.fd < 0) {
        free(cmdline_raw);
        return -ENOENT;
    }

    cmd_cursor = exchnd_fgets(cmdline_raw,
                              EXCHND_CMDLINE_LEN,
                              &file);
    exchnd_close(&file);

    collect_stat(0, pid_str, s_data);

    /* User space process: Add command line */
    if (cmd_cursor) {
        /* cmdline separates the parameters with '\0', replace with ' ' */
        while (len < EXCHND_CMDLINE_LEN) {
            if (cmdline_raw[len] == '\0') {
                cmdline_raw[len] = ' ';

                if ((len + 1 >= EXCHND_CMDLINE_LEN) ||
                    (cmdline_raw[len + 1] == '\0'))
                    break;
            }

            len++;
        }
    } else {
        /* Kernel process: Add executable name */
        memmove(s_data[STAT_COMM] + 1,
                s_data[STAT_COMM],
                strlen(s_data[STAT_COMM]));
        s_data[STAT_COMM][0] = '[';
        s_data[STAT_COMM][strlen(s_data[STAT_COMM])] = ']';
    }

    switch (type) {
    case PROCESS_LINE_FULL:
    {
        unsigned long swapents = collect_swapents(pid_str);

        snprintf(buffer, size,
                 TMPL_PROCESS_LIST,
                 s_data[STAT_PID],
                 s_data[STAT_STATE],
                 s_data[STAT_PRIORITY],
                 s_data[STAT_NICE],
                 s_data[STAT_RTPRIORITY],
                 s_data[STAT_POLICY],
                 s_data[STAT_RSS],
                 swapents,
                 s_data[STAT_COMM],
                 cmdline_raw);
        break;
    }
    case PROCESS_LINE_THREAD:
    {
        snprintf(buffer, size,
                 TMPL_THREAD_LIST,
                 s_data[STAT_PID],
                 s_data[STAT_PGRP],
                 s_data[STAT_STATE],
                 s_data[STAT_COMM]);
        break;
    }
    default:

        if (cmd_cursor)
            snprintf(buffer, size, "%s", cmdline_raw);
        else
            snprintf(buffer, size, "%s", s_data[STAT_COMM]);

        break;
    }

    free(cmdline_raw);
    return 0;
}

/*
 * \func get_indent_string
 *
 * This function writes spaces to a provided character buffer regarding to
 * the indent level.
 *
 * \param buf The buffer to write to
 * \param level The indent level
 *
 * \return None
 */
static void get_indent_string(char *buf, int level)
{
    int i;
    buf[0] = '\0';

    for (i = 0; i < level; i++)
        strncat(buf, STR_INDENT, sizeof(STR_INDENT));
}

/*
 * \func user_hz_to_ns
 *
 * This function converts a value in the USER_HZ unit into nanoseconds.
 *
 * \param val A value in USER_HZ
 *
 * \return The value in nanoseconds
 */
static unsigned long long user_hz_to_ns(long val)
{
    unsigned long long user_hz;

    user_hz = sysconf(_SC_CLK_TCK);

    return val * (NS_PER_S / user_hz);
}

/*
 * \func cg_get_subsystem_path
 *
 * This function gets the mount point of a specific cgroups sub system.
 *
 * \param subcomp_name The name of the sub system
 * \param path_name The buffer to write the mount path to.
 *
 * \return 0 on success, negative error code otherwise.
 */
static int cg_get_subsystem_path(char *subcomp_name, char *path_name)
{
    struct exchnd_file f_mounts;
    char *filename = MOUNTS;
    char line[128];
    int rc = -EINVAL;

    exchnd_init_file(&f_mounts);
    f_mounts.fd = open(filename, O_RDONLY);

    if (f_mounts.fd > 0) {
        while (exchnd_fgets(line, sizeof(line), &f_mounts) != NULL)
            if (strstr(line, subcomp_name)) {
                sscanf(line, "%*s %s", path_name);
                rc = 0;
                break;
            }

        exchnd_close(&f_mounts);
    }

    return rc;
}

/*
 * \func cg_has_procs
 *
 * This function checks if the cgroup.procs file has any process IDs listed.
 *
 * \param path The path of the hierarchy entry to check.
 *
 * \return 1 if cgroup.procs has any entries, 0 otherwise.
 */
static int cg_has_procs(char *path)
{
    int f_proc;
    char filename[PATH_MAX];
    char line = '\0';
    int has_procs = 0;

    snprintf(filename, sizeof(filename), "%s/%s", path, CG_PROCS);
    f_proc = open(filename, O_RDONLY);

    if (f_proc > 0) {
        int ret = read(f_proc, &line, 1);

        if ((ret > 0) && (line != '\0'))
            has_procs = 1;

        close(f_proc);
    }

    return has_procs;
}

/*
 * \func cg_read_procs
 *
 * This function read the process IDs of a cgroup entry.
 *
 * \param procs_filename The filename of the .procs file
 * \param buffer A character buffer to write the values to
 * \param bufsize Size of the buffer
 *
 * \return 0
 */
static int cg_read_procs(char *procs_filename, char *buffer, int bufsize)
{
    struct exchnd_file f_procs;
    char line[32];
    int len = 0;
    buffer[0] = '\0';

    exchnd_init_file(&f_procs);
    f_procs.fd = open(procs_filename, O_RDONLY);

    /* klockwork fix */
    if (f_procs.fd >= 0) {
        while (exchnd_fgets(line, sizeof(line), &f_procs) != NULL) {
            len = strlen(line);
            line[len - 1] = '\0';

            strncat(buffer, line, bufsize);
            bufsize -= len;

            if (bufsize <= 0)
                break;
        }

        exchnd_close(&f_procs);
    }

    return 0;
}

/*
 * \func cg_cpuacct_get_usage
 *
 * This function gets the total CPU time (in nanoseconds) consumed by all
 * tasks in this cgroup (including tasks lower in the hierarchy).
 *
 * \param path The path of the cgroup
 *
 * \return The CPU time in nanoseconds
 */
static unsigned long long cg_cpuacct_get_usage(char *path)
{
    struct exchnd_file f_total;
    unsigned long long total = 0;
    char filename[NAME_MAX];
    char line[128];
    char *line_cursor;
    char *end;

    snprintf(filename, sizeof(filename), "%s/cpuacct.usage", path);

    exchnd_init_file(&f_total);
    f_total.fd = open(filename, O_RDONLY);

    /* klockwork fix */
    if (f_total.fd >= 0) {
        line_cursor = exchnd_fgets(line, sizeof(line), &f_total);

        if (line_cursor == NULL) {
            printf("Error reading file %s\n", filename);
        } else {
            total = strtoull(line_cursor, &end, BASE_DEC);

            if ((total == 0) && (end == line)) {
                printf("Error converting value %s\n", line);
            } else if (total == ULLONG_MAX) {
                printf("Error converting, overflow %s\n", line);
                total = 0;
            }
        }

        exchnd_close(&f_total);
    } else {
        printf("Error opening file %s\n", filename);
    }

    return total;
}

/*
 * \func cg_cpuacct_get_user_system
 *
 * This function gets the user and system time from a cgroup, recoded by the
 * CPU Accounting.
 *
 * \param path The cgroup path
 * \param user A pointer to the value where the user time shall be stored
 * \param sy A pointer to the value where the system time shall be stored
 *
 * \return 0 on success, a negative error number otherwise.
 */
static int cg_cpuacct_get_user_system(char *path,
                                      unsigned long long *user,
                                      unsigned long long *sy)
{
    long user_ticks = 0;
    long system_ticks = 0;
    struct exchnd_file f_stat;
    char filename[PATH_MAX];
    char line[128];
    char *line_cursor;

    snprintf(filename, sizeof(filename), "%s/cpuacct.stat", path);
    exchnd_init_file(&f_stat);
    f_stat.fd = open(filename, O_RDONLY);

    if (f_stat.fd > 0) {
        line_cursor = exchnd_fgets(line, sizeof(line), &f_stat);

        if (line_cursor == line)
            sscanf(line, "%*s %ld", &user_ticks);

        line_cursor = exchnd_fgets(line, sizeof(line), &f_stat);

        if (line_cursor == line)
            sscanf(line, "%*s %ld", &system_ticks);

        exchnd_close(&f_stat);
    }

    *user = user_hz_to_ns(user_ticks);
    *sy = user_hz_to_ns(system_ticks);

    return 0;
}

/*
 * \func cg_cpuacct_get_per_cpu
 *
 * This function gets the CPU usage of a process per CPU, recorded by the
 * cgroups CPU Accounting subsystem.
 *
 * \param path The path to the cgroup
 * \param vals A character buffer to write the values to
 * \param vals_len The length of the character buffer
 *
 * \return 0 on success, a negative error number otherwise.
 */
static int cg_cpuacct_get_per_cpu(char *path, char *vals, int vals_len)
{
    struct exchnd_file f_stat;
    char filename[PATH_MAX];
    char *line_cursor;
    int rc = 0;

    snprintf(filename, sizeof(filename), TMPL_CG_PERCPU, path);
    exchnd_init_file(&f_stat);
    f_stat.fd = open(filename, O_RDONLY);

    /* klockwork fix */
    if (f_stat.fd >= 0) {
        line_cursor = exchnd_fgets(vals, vals_len, &f_stat);

        if (line_cursor != vals)
            rc = -EINVAL;

        exchnd_close(&f_stat);
    } else {
        rc = -EINVAL;
    }

    remove_trailing_newline(vals);

    if (rc != 0)
        snprintf(vals, vals_len, "error");

    return rc;
}

/*
 * \func cg_cpuacct_gen_entry
 *
 * This function generates the output for the CPU Accounting subsystem.
 *
 * \param ent Pointer to the FTSENT structure
 *
 * \return 1 if cgroup.procs has any entries, 0 otherwise.
 */
static int cg_cpuacct_gen_entry(FTSENT *ent)
{
    char indent[32];
    char percpu[128];
    char filename[PATH_MAX];
    unsigned long long total;
    unsigned long long user;
    unsigned long long sy;
    char buffer[512];
    int bufsize = sizeof(buffer);

    if (ent->fts_info == FTS_D) {
        get_indent_string(indent, ent->fts_level);
        total = cg_cpuacct_get_usage(ent->fts_path);
        cg_cpuacct_get_user_system(ent->fts_path, &user, &sy);
        cg_cpuacct_get_per_cpu(ent->fts_path, percpu, sizeof(percpu));

        snprintf(buffer, bufsize, "%s- %s\n"
                 "%s  total:  %llu\n"
                 "%s  user:   %llu\n"
                 "%s  system: %llu\n"
                 "%s  perCPU: %s",
                 indent, ent->fts_name,
                 indent, total,
                 indent, user,
                 indent, sy,
                 indent, percpu);

        exchnd_backend_store(buffer, strlen(buffer));

        if (ent->fts_level >= 2) {
            snprintf(buffer, bufsize, "%s  PIDs:   ", indent);
            snprintf(filename, sizeof(filename), "%s/%s", ent->fts_path,
                     CG_PROCS);
            cg_read_procs(filename, buffer + strlen(buffer), bufsize -
                          strlen(buffer));
            exchnd_backend_store(buffer, strlen(buffer));
        }

        return 1;
    }

    return 0;
}

/*
 * \func cg_memory_get_max_usage
 *
 * This function reads the recorded maximum memory usage of a cgroup hierarchy
 * entry into a provided buffer.
 *
 * \param path The hierarchy entry to process
 * \param val The buffer to write the value into
 * \param valsize The size of the buffer (in bytes)
 *
 * \return 0 on success, negative error code otherwise.
 */
static int cg_memory_get_max_usage(char *path, char *val, int valsize)
{
    struct exchnd_file f_usage;
    char filename[NAME_MAX];
    int rc = -EINVAL;

    snprintf(filename, sizeof(filename), "%s/%s", path, CG_MEM_MAX_USAGE);
    exchnd_init_file(&f_usage);
    f_usage.fd = open(filename, O_RDONLY);

    if (f_usage.fd > 0) {
        if (exchnd_fgets(val, valsize, &f_usage) != NULL) {
            remove_trailing_newline(val);
            rc = 0;
        }

        exchnd_close(&f_usage);
    } else {
        rc = -ENOENT;
    }

    if (rc != 0)
        snprintf(val, valsize, "error");

    return rc;
}

/*
 * \func cg_memory_get_usage
 *
 * This function reads the memory usage of a cgroup hierarchy entry into a
 * provided buffer.
 *
 * \param path The hierarchy entry to process
 * \param val The buffer to write the value into
 * \param valsize The size of the buffer (in bytes)
 *
 * \return 0 on success, negative error code otherwise.
 */
static int cg_memory_get_usage(char *path, char *val, int valsize)
{
    struct exchnd_file f_usage;
    char filename[NAME_MAX];
    int rc = -1;

    snprintf(filename, sizeof(filename), "%s/%s", path, CG_MEM_USAGE);
    exchnd_init_file(&f_usage);
    f_usage.fd = open(filename, O_RDONLY);

    /* klockwork fix */
    if (f_usage.fd >= 0) {
        if (exchnd_fgets(val, valsize, &f_usage) != NULL) {
            remove_trailing_newline(val);
            rc = 0;
        }

        exchnd_close(&f_usage);
    } else {
        rc = -ENOENT;
    }

    return rc;
}

static int cg_memory_get_stat(char *path, unsigned long long *cache,
                              unsigned long long *rss, unsigned long long *swap)
{
    struct exchnd_file f_stat;
    char filename[NAME_MAX];
    char line[128];
    char label[128];
    unsigned long long val;
    int items_parsed = 0;
    int rc = -ENOENT;

    snprintf(filename, sizeof(filename), "%s/%s", path, CG_MEM_STAT);
    exchnd_init_file(&f_stat);
    f_stat.fd = open(filename, O_RDONLY);

    /* klockwork fix */
    if (f_stat.fd >= 0) {
        while (exchnd_fgets(line, sizeof(line), &f_stat) != NULL) {
            /* Security of sscanf: Format of memory.stat is <label> <value> */
            items_parsed = sscanf(line, "%s %llu", label, &val);

            if (items_parsed != 2) {
                rc = -EINVAL;
                break;
            }

            if ((cache != NULL) &&
                (strncmp(label, MEMSTAT_CACHE, sizeof(MEMSTAT_CACHE)) == 0))
                *cache = val;
            else if ((rss != NULL) &&
                     (strncmp(label, MEMSTAT_RSS, sizeof(MEMSTAT_RSS)) == 0))
                *rss = val;
            else if ((swap != NULL) &&
                     (strncmp(label, MEMSTAT_SWAP, sizeof(MEMSTAT_SWAP)) == 0))
                *swap = val;
        }

        exchnd_close(&f_stat);
        rc = EXIT_SUCCESS;
    }

    return rc;
}

/*
 * \func cg_memory_gen_entry
 *
 * This function processes a cgroups subsystem directory tree and calls a
 * provided function for every directory entry.
 *
 * \param walk_path The directory to be processed
 * \param gen_entry The function that is called for every directory entry
 *
 * \return None
 */
static int cg_memory_gen_entry(FTSENT *ent)
{
    char indent[32];
    char usage[64];
    char max_usage[64];
    unsigned long long cache = 0;
    unsigned long long rss = 0;
    char buffer[512];
    char filename[PATH_MAX];
    int bufsize = sizeof(buffer);

    if (ent->fts_info == FTS_D) {
        get_indent_string(indent, ent->fts_level);

        if (cg_has_procs(ent->fts_path) == 1) {
            cg_memory_get_usage(ent->fts_path, usage, sizeof(usage));
            cg_memory_get_stat(ent->fts_path, &cache, &rss, NULL);
            cg_memory_get_max_usage(ent->fts_path, max_usage,
                                    sizeof(max_usage));

            snprintf(buffer, bufsize, "%s- %s\n"
                     "%s  usage:     %s\n"
                     "%s  cache:     %llu\n"
                     "%s  rss:       %llu\n"
                     "%s  max usage: %s",
                     indent, ent->fts_name,
                     indent, usage,
                     indent, cache,
                     indent, rss,
                     indent, max_usage);
            exchnd_backend_store(buffer, strlen(buffer));
        }

        if (ent->fts_level >= 2) {
            snprintf(buffer, bufsize, "%s  PIDs:      ", indent);
            snprintf(filename, sizeof(filename), "%s/%s", ent->fts_path,
                     CG_PROCS);
            cg_read_procs(filename, buffer + strlen(buffer), bufsize -
                          strlen(buffer));
            exchnd_backend_store(buffer, strlen(buffer));
        }

        return 1;
    }

    return 0;
}

/*
 * \func cg_process_hierarchy
 *
 * This function processes a cgroups subsystem directory tree and calls a
 * provided function for every directory entry.
 *
 * \param walk_path The directory to be processed
 * \param gen_entry The function that is called for every directory entry
 *
 * \return None
 */
static void cg_process_hierarchy(char *walk_path, int (*gen_entry)(FTSENT *))
{
    FTS *fts;
    FTSENT *ent;
    FTSENT *child_ent;
    int fts_options = FTS_COMFOLLOW | FTS_LOGICAL | FTS_NOCHDIR;
    char *paths[] = {
        walk_path,
        NULL
    };

    fts = fts_open(paths, fts_options, NULL);

    if (fts == NULL) {
        exchnd_log_error("Error walking cgroups hierarchy %s",
                         walk_path);
        /* Error has been logged, exit */
        return;
    }

    child_ent = fts_children(fts, 0);

    if (child_ent == NULL)
        /* Nothing to do here, exit */
        return;

    while ((ent = fts_read(fts)) != NULL)
        if (gen_entry(ent))
            /* Found */
            break;

    fts_close(fts);
}

/*
 * \func cg_collect_cpuacct
 *
 * This function triggers the generation of the cgroups status of the CPU
 * Accounting subsystem if it is enabled.
 *
 * \return None
 */
static void cg_collect_cpuacct()
{
    char path[PATH_MAX];

    if (cg_get_subsystem_path(CG_SUB_CPUACCT, path) == 0)
        cg_process_hierarchy(path, cg_cpuacct_gen_entry);
}

/*
 * \func cg_collect_memory
 *
 * This function triggers the generation of the cgroups status of the memory
 * subsystem if it is enabled.
 *
 * \return None
 */
static void cg_collect_memory()
{
    char path[PATH_MAX];

    if (cg_get_subsystem_path(CG_SUB_MEMORY, path) == 0)
        cg_process_hierarchy(path, cg_memory_gen_entry);
}

/*
 * \func get_fs_state
 *
 * This function writes the file system status into a provided buffer.
 *
 * \param path Mount point or any file of the file system
 * \param buffer Buffer to store the status information
 * \param bufsize Size of the buffer
 *
 * \return 0 if successful. An error code otherwise
 */
static int get_fs_state(char *path, char *buffer, size_t bufsize)
{
    struct statvfs st;
    int rc;
    unsigned int written;

    rc = statvfs(path, &st);

    if (rc != 0)
        snprintf(buffer, bufsize, "Error getting file system state: %s",
                 strerror(errno));

    if (st.f_blocks > 0) {
        written = snprintf(buffer, bufsize, "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n"
                           "%-20s %lu\n",
                           "Block size:", st.f_bsize,
                           "Fragment size:", st.f_frsize,
                           "Fragments:", st.f_blocks,
                           "Free blocks:", st.f_bfree,
                           "Available blocks:", st.f_bavail,
                           "Inodes:", st.f_files,
                           "Free inodes:", st.f_ffree,
                           "Available inodes:", st.f_favail,
                           "File system ID:", st.f_fsid,
                           "Mount flags:", st.f_flag,
                           "Max filename length:", st.f_namemax);

        /* Provided buffer is too small */
        if (written >= bufsize)
            rc = -ENOMEM;
    }

    return rc;
}

/*
 * \func log_backtrace
 *
 * Generic function to be used by backtrace and backtrace_all_threads
 * collectors.
 * This function will prepare the libunwind element and call the unwinding
 * function.
 *
 * \param pid PID of the process to take the backtrace of.
 * \param extend Shall the backtrace be extended to provide arguments and
 *        memory dump.
 * \param eunw Structure containing relevant unwinding and memory map data.
 */
int log_backtrace(int pid, int extend, struct exchnd_unwind *eunw)
{
    char stat_raw[512];
    int result = 0;

    struct exchnd_file file;
    char filename[32];
    char thread_name[PATH_MAX];
    char *buf_head;

    buf_head = &thread_name[0];
    snprintf(buf_head, sizeof(thread_name), "%d: ", pid);

    buf_head += strlen(buf_head);
    /* Extract comm from /proc/<pid>/stat */
    snprintf(filename, sizeof(filename), TMPL_STAT, pid);

    exchnd_init_file(&file);
    file.fd = open(filename, O_RDONLY);

    if (file.fd < 0)
        return -ENOENT;

    /* Getting executable from stat [pid (exec_name) status ...]*/
    if (exchnd_fgets(stat_raw, 512, &file))
        get_comm(stat_raw, buf_head);

    exchnd_close(&file);

    exchnd_backend_store(thread_name, strlen(thread_name));

    if (extend != EXCHND_FB_GUESS)
        result = __log_backtrace(pid, extend, eunw);
    else
        result = get_func_from_stack(pid, eunw);

    return result;
}

/*
 * \func collector_backtrace_all_threads
 *
 * This collector handles the generation of a backtrace for all threads of a
 * user space process.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_backtrace_all_threads(ExchndInterface_t *exh_if,
                                    struct exchnd_message_header *header,
                                    char *data)
{
    DIR *dir;
    struct dirent *ent;
    char filename[32];
    long thread_pid;
    int rc = 0;

    struct exchnd_unwind eunw = { 0 };

    /* let satisfy lint. */
    (void)data;

    int idx = 0;
    idx_max = 0;

    exchnd_backend_store(LABEL_BACKTRACE_ALL_THREADS,
                         strlen(LABEL_BACKTRACE_ALL_THREADS));

    snprintf(filename, sizeof(filename), TMPL_TASK, header->pid);
    dir = opendir(filename);

    if (dir == NULL) {
        exchnd_log_error("Error opening %s", filename);
        rc = -EIO;
    } else {
        eunw.num_maps = get_maps(header->pid, &eunw.maps);

        while (rc >= 0) {
            ent = readdir(dir);

            /* End of the directory stream */
            if (ent == NULL)
                break;

            /* PIDs start with numbers... */
            if (!isdigit(ent->d_name[0]))
                continue;

            if (idx_max >= MAX_THREADS) {
                exchnd_log_error("Too much threads.");
                break;
            }

            thread_pid = atol(ent->d_name);
            pt_group_attach(exh_if, header, thread_pid);
        }

        closedir(dir);

        while (idx < idx_max) {
            exchnd_touch_wd(0, EHM_BACKTRACE_ALL_THREADS);
            log_backtrace(attached[idx], 0, &eunw);
            idx++;
        }

        free(eunw.maps);
    }

    return rc;
}

/*
 * \func collector_backtrace
 *
 * This collector handles the generation of a backtrace for a user space
 * process.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_backtrace(ExchndInterface_t *exh_if,
                        struct exchnd_message_header *header,
                        char *data)
{
    int extend = 0;

    struct exchnd_unwind eunw = { 0 };
    int ret = 0;

    /* let satisfy lint. */
    (void)data;

    exchnd_backend_store(LABEL_BACKTRACE, strlen(LABEL_BACKTRACE));
#if (EXCHND_VERSION > 0x206)

    if (((action & EXCHND_EXT_BT) && (header->sig & (1 << SIGABRT))) ||
        (action & EXCHND_EXT_BT_ALL)) {
        char buffer[] =
            "Up to " STRINGIFY(ARG_CNT) " potential arguments shown"
            " (last may be variables and/or\n"
            "artifacts from previous code execution):\n";
        exchnd_backend_store(buffer, strlen(buffer));

        extend = 1;
    }

#endif

    if (exh_if->PTRACE_attached == EXH_DETACHED) {
        int mustkill = (action & EXCHND_KILL);

        if (pt_attach(header->pid, mustkill) != 0)
            return -1;

        exh_if->PTRACE_attached = EXH_THREAD;
    }

    eunw.num_maps = get_maps(header->pid, &eunw.maps);

    ret = log_backtrace(header->pid, extend, &eunw);

    if ((ret != EXCHND_FB_NONE) && (action & EXCHND_FRAME_BT_FALLBACK)) {
        char buffer[] = "Fallen back to frame unwinding due to error.";
        exchnd_backend_store(LABEL_FB_BACKTRACE, strlen(LABEL_FB_BACKTRACE));
        exchnd_backend_store(buffer, strlen(buffer));
        ret = log_backtrace(header->pid, EXCHND_FB_FRAME, &eunw);
    }

    if (!extend && (action & EXCHND_EXT_BT_FALLBACK) &&
        (ret != EXCHND_FB_NONE)) {
        char buffer[] = "---/!\\/!\\/!\\---\n"
            "Fallen back to extended backtrace due to error"
            " and hope for more information.\n"
            "---/!\\/!\\/!\\---\n";
        char buffer1[] = "Backtrace with indirect dump of potentials"
            " arguments and variables:\n";

        exchnd_backend_store(LABEL_EXT_BACKTRACE, strlen(LABEL_EXT_BACKTRACE));
        exchnd_backend_store(buffer, strlen(buffer));
        exchnd_backend_store(buffer1, strlen(buffer1));

        ret = log_backtrace(header->pid, EXCHND_FB_EXTENDED, &eunw);
    }

    if ((action & EXCHND_GUESS_FALLBACK) && (ret != EXCHND_FB_NONE)) {
        char buffer[] = "---/!\\/!\\/!\\---\n"
            "Fail to get proper backtrace. Getting functions from stack.\n"
            "---/!\\/!\\/!\\---\n";

        exchnd_backend_store(LABEL_GUESS_BACKTRACE,
                             strlen(LABEL_GUESS_BACKTRACE));
        exchnd_backend_store(buffer, strlen(buffer));

        ret = log_backtrace(header->pid, EXCHND_FB_GUESS, &eunw);
    }

    free(eunw.maps);

    return ret;
}

/*
 * \func collector_cgroups
 *
 * This collector handles the generation of the cgroups status
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_cgroups(ExchndInterface_t *exh_if,
                      struct exchnd_message_header *header,
                      char *data)
{
    struct exchnd_file f_cgroups;
    char line[128];
    char subsystem_name[64];
    int rc = 0;

    /* let satisfy lint. */
    (void)data;
    (void)header;
    (void)exh_if;

    exchnd_backend_store(LABEL_CG_STATE,
                         strlen(LABEL_CG_STATE));
    exchnd_init_file(&f_cgroups);
    f_cgroups.fd = open(CG_INFO_FILE, O_RDONLY);

    /* klockwork fix */
    if (f_cgroups.fd >= 0) {
        /* Loop through all cgroups sub systems */
        while (exchnd_fgets(line, sizeof(line), &f_cgroups) != NULL) {
            /* And execute collector functions for specific ones */
            sscanf(line, "%s ", subsystem_name);

            if (strncmp(subsystem_name, CG_SUB_CPUACCT,
                        sizeof(CG_SUB_CPUACCT)) == 0)
                cg_collect_cpuacct();

            if (strncmp(subsystem_name, CG_SUB_MEMORY,
                        sizeof(CG_SUB_MEMORY)) == 0)
                cg_collect_memory();
        }

        exchnd_close(&f_cgroups);
    } else {
        exchnd_log_error("Error locating cgroups\n");
        rc = -ENOENT;
    }

    return rc;

}

/*
 * \func collector_fs_state
 *
 * This collector handles the generation of a fs_state for a user space
 * process.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_fs_state(ExchndInterface_t *exh_if,
                       struct exchnd_message_header *header,
                       char *data)
{
    char buffer[MAX_MSG_LEN];
    char device[256];
    char mountpoint[256];
    char filesystem[32];
    struct exchnd_file file;
    unsigned int i;
    int ignore_entry;
    int rc = -EINVAL;
    char *ignore_list[] = {
        "/dev",
        "/proc",
        "/sys"
    };

    /* let satisfy lint. */
    (void)data;
    (void)header;
    (void)exh_if;

    exchnd_backend_store(LABEL_FS_STATE,
                         strlen(LABEL_FS_STATE));
    exchnd_init_file(&file);
    file.fd = open(MOUNTS, O_RDONLY);

    /* klockwork fix */
    if (file.fd >= 0) {
        while (exchnd_fgets(buffer, sizeof(buffer), &file) != 0) {
            sscanf(buffer, "%s %s %s", device, mountpoint, filesystem);
            ignore_entry = 0;

            /* Check if mount point is in the ignore list */
            for (i = 0; i < sizeof(ignore_list) / sizeof(char *); i++)
                /* Beginning matches entry in the ignore list */
                if (strncmp(ignore_list[i], mountpoint,
                            strlen(ignore_list[i])) == 0)
                    ignore_entry = 1;

            if (ignore_entry == 0) {
                snprintf(buffer, sizeof(buffer),
                         "%-20s %s\n"
                         "%-20s %s\n"
                         "%-20s %s",
                         "Mount point:", mountpoint,
                         "Type:", filesystem,
                         "Device:", device);
                exchnd_backend_store(buffer, strlen(buffer));
                rc = get_fs_state(mountpoint, buffer, sizeof(buffer));

                if (rc == 0)
                    exchnd_backend_store(buffer,
                                         strlen(buffer));
                else
                    exchnd_log_error("Error getting fs info");
            }
        }

        exchnd_close(&file);
    }

    return rc;
}

/*
 * \func collector_memory_map
 *
 * This collector handles the generation of a memory map of a user space
 * process. /proc/<PID/maps is used as data source.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_memory_map(ExchndInterface_t *exh_if,
                         struct exchnd_message_header *header,
                         char *data)
{
    char filename[32];
    struct exchnd_file file;
    char buffer[MAX_MSG_LEN];
    int rc = 0;

    (void)exh_if;
    (void)data;

    snprintf(filename, sizeof(filename), TMPL_MAPS, header->pid);
    snprintf(buffer, sizeof(buffer), LABEL_MEMORYMAP);
    exchnd_backend_store(buffer, strlen(buffer));

    exchnd_init_file(&file);
    file.fd = open(filename, O_RDONLY);

    if (file.fd < 0) {
        exchnd_log_error("Error opening file: %s", filename);
        return -EIO;
    }

    while (exchnd_fgets(buffer, sizeof(buffer), &file)) {
        remove_trailing_newline(buffer);
        exchnd_backend_store(buffer, strlen(buffer));
    }

    exchnd_close(&file);

    return rc;
}

/*
 * \func collector_mem_dump
 *
 * This collector handles the generation of a memory dump for a user space
 * process.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_mem_dump(ExchndInterface_t *exh_if,
                       struct exchnd_message_header *header,
                       char *data)
{
    unsigned long faulty_add = 0;
    int ret = 0;

    /* We need internal data to proceed. */
    /* If not available, just skip this module. */
    if (header->flags.internal == 0)
        return ret;

    exchnd_backend_store(LABEL_MEM_DUMP, strlen(LABEL_MEM_DUMP));

    faulty_add = strtoul((char *)data, NULL, BASE_HEX);

    if (faulty_add)
        ret = __collector_mem_dump(exh_if, header, faulty_add);

    return ret;
}

/*
 * \func collector_process_list
 *
 * This collector handles the generation of the process list.
 * The list includes all running processes. Processes with multiple threads
 * are shown as one process.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_process_list(ExchndInterface_t *exh_if,
                           struct exchnd_message_header *header,
                           char *data)
{
    DIR *dir;
    struct dirent *ent;
    int rc = 0;
    char buffer[MAX_MSG_LEN];

    (void)exh_if;
    (void)header;
    (void)data;

    exchnd_backend_store(LABEL_PROCESS_LIST, strlen(LABEL_PROCESS_LIST));
    exchnd_backend_store(TABLE_PROCESS_LIST, strlen(TABLE_PROCESS_LIST));

    dir = opendir(PROCFS);

    if (dir == NULL) {
        exchnd_log_error("Error opening procfs");
        rc = -EIO;
    } else {
        while (rc == 0) {
            ent = readdir(dir);

            /* End of the directory stream */
            if (ent == NULL)
                break;

            /* PIDs start with numbers... */
            if (isdigit(ent->d_name[0])) {
                get_process_line(ent->d_name,
                                 buffer,
                                 MAX_MSG_LEN,
                                 PROCESS_LINE_FULL);
                exchnd_backend_store(buffer, strlen(buffer));
            }
        }

        closedir(dir);
    }

    return rc;
}

/*
 * \func collector_stack_dump
 *
 * This collector handles the generation of a stack dump for a user space
 * process.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_stack_dump(ExchndInterface_t *exh_if,
                         struct exchnd_message_header *header,
                         char *data)
{
    /* We need internal data to proceed. */
    /* If not available, just skip this module. */
    if (header->flags.internal == 0)
        return 0;

    exchnd_backend_store(LABEL_STACK_DUMP, strlen(LABEL_STACK_DUMP));

    return __collector_stack_dump(exh_if, header, data);
}

/*
 * \func collector_thread_list
 *
 * This collector generates a list of all the threads of a given process.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_thread_list(ExchndInterface_t *exh_if,
                          struct exchnd_message_header *header,
                          char *data)
{
    DIR *dir;
    struct dirent *ent;
    char buffer[MAX_MSG_LEN];
    char filename[32];
    int rc = 0;

    (void)exh_if;
    (void)data;

    exchnd_backend_store(LABEL_THREAD_LIST,
                         strlen(LABEL_THREAD_LIST));
    exchnd_backend_store(TABLE_THREAD_LIST, strlen(TABLE_THREAD_LIST));

    snprintf(filename, sizeof(filename), TMPL_TASK, header->pid);
    dir = opendir(filename);

    if (dir == NULL) {
        exchnd_log_error("Error opening %s", filename);
        rc = -EIO;
    } else {
        while (rc == 0) {
            ent = readdir(dir);

            /* End of the directory stream */
            if (ent == NULL)
                break;

            /* PIDs start with numbers... */
            if (isdigit(ent->d_name[0])) {
                get_process_line(ent->d_name,
                                 buffer,
                                 MAX_MSG_LEN,
                                 PROCESS_LINE_THREAD);
                exchnd_backend_store(buffer, strlen(buffer));
            }
        }

        closedir(dir);
    }

    return rc;
}

/*
 * \func collector_application_specific
 *
 * This collector will not collect something. It will only call an application
 * specific function after loading a specific library.
 * The daemon shall be already attached.
 */
int collector_application_specific(ExchndInterface_t *exh_if,
                                   struct exchnd_message_header *header,
                                   char *data)
{
    char *libName = data;
    void *handle = NULL;
    int (*func)(void (*store)(char *msg, int len), pid_t);
    char *error;
    int ret = EXIT_FAILURE;

    exchnd_backend_store(LABEL_APPLICATION_SPECIFIC,
                         strlen(LABEL_APPLICATION_SPECIFIC));

    /* We need internal data to proceed. */
    /* If not available, just skip this module. */
    if (header->flags.internal == 0) {
        exchnd_log_error("AS1: No internal data");
        return ret;
    }

    /* Attach only if it is not already attached */
    if (exh_if->PTRACE_attached == EXH_DETACHED) {
        int mustkill = (action & EXCHND_KILL);

        if (pt_attach(header->pid, mustkill) != 0)
            return ret;

        exh_if->PTRACE_attached = EXH_THREAD;
    }

    handle = dlopen(libName, RTLD_NOW);

    if (!handle) {
        exchnd_log_error("AS1: Unable do open shared library (%s)",
                         dlerror());
        return ret;
    }

    dlerror(); /* Clear any existing error */
    func = (void *)dlsym(handle, "exchnd_application_specific");

    error = dlerror();

    if (error != NULL) {
        exchnd_log_error("AS1: Unable to find function symbol (%s)",
                         error);
        dlclose(handle);
        return ret;
    }

    ret = func(exchnd_backend_store, header->pid);
    dlclose(handle);
    /* Execute function. */
    return ret;
}

/*
 * \func __collector_cpu_usage
 *
 * Collect cpu usage for both processes and threads.
 *
 * \param pid  The pid of the thread to collect the information of. 0 if it's a
 *             process.
 *
 * \return 0 if successful. An error code otherwise
 */
int __collector_cpu_usage(int pid)
{
    DIR *dir;
    struct dirent *ent;
    int rc = 0;
    char buffer[MAX_MSG_LEN];
    char *s = buffer;

    if (!pid) {
        exchnd_backend_store(LABEL_CPU_USAGE, strlen(LABEL_CPU_USAGE));

        dir = opendir(PROCFS);
    } else {
        char filename[32];
        snprintf(filename, sizeof(filename), TMPL_TASK, pid);
        dir = opendir(filename);
    }

    if (dir == NULL) {
        exchnd_log_error("Error opening procfs");
        rc = -EIO;
    } else {
        while (rc == 0) {
            ent = readdir(dir);

            /* End of the directory stream */
            if (ent == NULL)
                break;

            /* PIDs start with numbers... */
            if (isdigit(ent->d_name[0]) && (atoi(ent->d_name) != pid)) {
                memset(buffer, '\0', MAX_MSG_LEN);
                s = buffer;

                if (pid) {
                    buffer[0] = '\t';
                    buffer[1] = '\\';
                    buffer[2] = '_';
                    buffer[3] = '_';
                    buffer[4] = '\t';
                    s += 4;
                }

                /* First get base information concerning process/thread*/
                /* PID State   cmdline*/
                get_cpu_time(pid, ent->d_name, s);
                exchnd_backend_store(buffer, strlen(buffer));

                /* pid != 0 means thread handling already ongoing */
                if (!pid)
                    /* Then collect info for each threads */
                    __collector_cpu_usage(atoi(ent->d_name));
            }
        }

        closedir(dir);
    }

    return rc;
}

/*
 * \func collector_cpu_usage
 *
 * This function collects information on cpu usage for each task running
 * on the system. That implies to run through the proc file system and read
 * stat information for each process and subprocess.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_cpu_usage(ExchndInterface_t *exh_if,
                        struct exchnd_message_header *header,
                        char *data)
{
    (void)exh_if;
    (void)header;
    (void)data;

    return __collector_cpu_usage(0);
}

static void get_app_path(char *cmdline)
{
    char *q = NULL;

    q = memchr(cmdline, ' ', EXCHND_CMDLINE_LEN);

    if (!q)
        return;

    *q = '\0';
}

static void notify_client(pid_t pid)
{
    /* Don't use calloc as it generate minor page faults */
    char *buf = calloc(1, EXCHND_CMDLINE_LEN);
    struct exchnd_file file = { 0 };
    char filename[32] = { 0 };
    char pid_str[32] = { 0 };

    snprintf(pid_str, 32, "%d", pid);

    if (!buf) {
        exchnd_log_error("Unable to allocate memory to notify clients.\n");
        return;
    }

    memset(buf, 0, EXCHND_CMDLINE_LEN);

    /* Read /proc/<pid>/cmdline */
    snprintf(filename, sizeof(filename), TMPL_CMDLINE, pid_str);
    exchnd_init_file(&file);
    file.fd = open(filename, O_RDONLY);

    if (file.fd < 0) {
        free(buf);
        exchnd_log_error("Unable to open cmdline file.\n");
        return;
    }

    exchnd_fgets(buf, EXCHND_CMDLINE_LEN, &file);
    exchnd_close(&file);

    get_app_path(buf);

    exchnd_server_add_msg(pid, buf);
}

/*
 * \func collector_action_start
 *
 * This collector handle the synchronization message from the exception handler
 * module. The message indicates that all messages for one exception are sent
 * and the module waits for an acknowledge that data is written.
 * The current implementation returns this acknowledge immediately because the
 * messages are handled sequentially.
 *
 * \param exh_if Interface to exception handler daemon instance
 * \param header Exception message header
 * \param data Exception message data
 *
 * \return 0 if successful. An error code otherwise
 */
int collector_action_start(ExchndInterface_t *exh_if,
                           struct exchnd_message_header *header,
                           char *data)
{
    pid_t ready = -1;
    int rc = 0;

    if (header->length != 1) {
        exchnd_log_error("Invalid length of start message length=%d",
                         header->length);
        rc = EINVAL;
        goto exit;
    }

    if (data[0] == '1') {
        exchnd_wait_default_backend();

        rc = ioctl(exh_if->efd, IOCTL_EXCHND_DAEMON_READY, &ready);

        if (rc != 0) {
            exchnd_log_error("ioctl IOCTL_EXCHND_DAEMON_READY failed: %s",
                             strerror(rc));
            rc = EIO;
            goto exit;
        }
    }

    if ((header->trigger == ET_PROCESS_SIGNAL) ||
        (header->trigger == ET_PROCESS_EXIT)) {
        if (exh_if->PTRACE_attached == EXH_DETACHED) {
            if (pt_attach(header->pid, (action & EXCHND_KILL)) != 0) {
                rc = EIO;
                goto exit;
            }

            exh_if->PTRACE_attached = EXH_THREAD;
        }

        if (action & EXCHND_SERVER)
            notify_client(header->pid);
    }

    if (!(action & EXCHND_KILL))
        rc = pt_detach(header->pid, exh_if->PTRACE_attached);

    /* If ESRCH the thread was probably already dead.
     * Just ignore it as there is nothing we can do.
     * Detach will be forced by the end of the tracing
     * thread.
     */
    if ((rc < 0) && (errno != ESRCH))
        exchnd_log_error("Restarting threads failed: %s",
                         strerror(errno));

exit:
    /* Reset flag independently from the ptrace result */
    exh_if->PTRACE_attached = EXH_DETACHED;

    /* Write end message */
    exchnd_log_error("***** end of exception log *****");

    return rc;
}

/**
 * \func exchnd_add_commandline
 *
 * Add the command line to the header of the messages if available.
 *
 * \param header Pointer to the header were information about the process
 *               can be found
 */
void exchnd_add_commandline(struct exchnd_message_header *header)
{
    char cmd[MAX_MSG_LEN] = { '\0' };
    char buffer[MAX_MSG_LEN] = { '\0' };
    char pid_str[32];

    snprintf(pid_str, 32, "%d", header->pid);

    get_process_line(pid_str, cmd, MAX_MSG_LEN, PROCESS_LINE_STARVED);

    snprintf(buffer, MAX_MSG_LEN, LABEL_CMDLINE, cmd);

    exchnd_backend_store(buffer, strlen(buffer));
}

collect collector_list[EHM_LAST_ELEMENT] = {
    [EHM_NONE] = NULL,
    [EHM_BACKTRACE] = collector_backtrace,
    [EHM_BACKTRACE_ALL_THREADS] = collector_backtrace_all_threads,
    [EHM_CGROUPS] = collector_cgroups,
    [EHM_CORE_DUMP] = NULL,
    [EHM_CPU_USAGE] = collector_cpu_usage,
    [EHM_FAULT_ADDRESS] = NULL,
    [EHM_FS_STATE] = collector_fs_state,
    [EHM_HIST_SYSCALLS] = NULL,
    [EHM_HIST_TASKSWITCHES] = NULL,
    [EHM_LRU_MEM_PAGES] = NULL,
    [EHM_MEMORY_MAP] = collector_memory_map,
    [EHM_MEMORY_DUMP] = collector_mem_dump,
    [EHM_MEMORY_USAGE] = NULL,
    [EHM_PROCESS_LIST] = collector_process_list,
    [EHM_PROCESSOR_REGISTERS] = NULL,
    [EHM_SCHED_INF] = NULL,
    [EHM_STACK_DUMP] = collector_stack_dump,
    [EHM_SYSTEM_INFO] = NULL,
    [EHM_THREAD_LIST] = collector_thread_list,
    [EHM_ACTION_START] = collector_action_start,
    [EHM_SYS_RESTART] = NULL,
    [EHM_APPLICATION_SPECIFIC1] = collector_application_specific,
    [EHM_APPLICATION_SPECIFIC2] = collector_application_specific,
    [EHM_APPLICATION_SPECIFIC3] = collector_application_specific,
    [EHM_APPLICATION_SPECIFIC4] = collector_application_specific,
    [EHM_APPLICATION_SPECIFIC5] = collector_application_specific
};
